using System;
using System.Collections;
using System.Drawing;
using System.Windows.Forms;
using System.Data;

using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using DivBrRules = gov.va.med.vbecs.Common.VbecsBrokenRules.Division;
using gov.va.med.vbecs.DAL.VAL;
using gov.va.med.vbecs.GUI;
using gov.va.med.vbecs.GUI.controls;
using InstitutionTable = gov.va.med.vbecs.Common.VbecsTables.AssociatedInstitution;
using DivTable = gov.va.med.vbecs.Common.VbecsTables.VamcDivision;


namespace gov.va.med.vbecs.admin.GUI
{	
	/// <summary>
	/// User control encapsulating UI functionality for editing division parameters. 
	/// </summary>
	public class EditDivisionParameters : BaseDivisionBoundControl
	{
		#region Division Comparers

		/// <summary>
		/// Standard comparer for <see cref="DivisionDefinition"/> objects comparing them by division code.
		/// Used to create a list of divisions sorted by code.
		/// </summary>
		private class DivisionDefinitionCodeComparer : BusinessObjectPropertyComparerBase
		{
			protected override int CompareProperties( object x, object y )
			{
				return Comparer.DefaultInvariant.Compare( ((DivisionDefinition)x).DivisionCode, ((DivisionDefinition)y).DivisionCode );
			}
		}

		/// <summary>
		/// Standard comparer for <see cref="DivisionDefinition"/> objects comparing them by division name.
		/// Used to create a list of divisions sorted by name.
		/// </summary>
		private class DivisionDefinitionNameComparer : BusinessObjectPropertyComparerBase
		{
			protected override int CompareProperties( object x, object y )
			{
				return Comparer.DefaultInvariant.Compare( ((DivisionDefinition)x).DivisionName, ((DivisionDefinition)y).DivisionName );
			}
		}
		
		#endregion

		#region DivisionListViewItem implementation
		/// <summary>
		/// The class represents list view item displaying data for a single 
		/// division in the list view at the bottom of the form.
		/// </summary>
		protected class DivisionListViewItem : BaseListViewItem
		{
			private enum DivSubItemIndex : int
			{
				Code = 1,
				Name,
				NationalFacilityName,
				ServiceType,
				AccessionAreaName,
				LockInactivityTimeout,	
				LabelPrinterComPort,
				LabelPrinterTcpPort,
				LabelPrinterIpAddress,
				PrinterName,							
				TimeZone,
				DaylightSavingsType,
				CustomDstStartDate,
				CustomDstEndDate
			};

			private static readonly int _numberOfSubItems;
			private static readonly Hashtable _allItems;			
			private readonly DivisionBase _division;			

			/// <summary>
			/// Static Constructor
			/// </summary>
			static DivisionListViewItem()
			{
				_numberOfSubItems = Enum.GetValues( typeof(DivSubItemIndex) ).Length;
				_allItems = new Hashtable();
			}			

			/// <summary>
			/// Constructor
			/// </summary>
			/// <param name="division"></param>
			public DivisionListViewItem( DivisionBase division ) 
				: base( _numberOfSubItems )
			{
				if( division == null )
					throw( new ArgumentNullException( "division" ) );

				this.UseItemStyleForSubItems = true;

				_division = division;				

				SyncWithDataObject();				

				lock( typeof( DivisionListViewItem ) )
					if( _allItems.ContainsKey( division ) )
						_allItems[ division ] = this;
					else
						_allItems.Add( division, this );
			}

			/// <summary>
			/// Division
			/// </summary>
			public DivisionBase Division
			{
				get
				{
					return _division;
				}
			}

			/// <summary>
			/// SyncWithDataObject
			/// </summary>
			public override void SyncWithDataObject()
			{				
				if( _division.IsActive != this.Checked ) // to prevent self-calling
					this.Checked = _division.IsActive;

				this.SubItems[ (int)DivSubItemIndex.Code ].Text = _division.DivisionCode;
				this.SubItems[ (int)DivSubItemIndex.Name ].Text = _division.DivisionName;
				this.SubItems[ (int)DivSubItemIndex.ServiceType ].Text = _division.ServiceType;

				this.SubItems[ (int)DivSubItemIndex.NationalFacilityName ].Text = ((FacilityDefinition)_nationalFacilities.GetByObjectID( _division.FacilityGuid ) ).FacilityName;

				this.SubItems[ (int)DivSubItemIndex.LockInactivityTimeout ].Text = _division.LockInactivityTimeoutMin.ToString();

				this.SubItems[ (int)DivSubItemIndex.AccessionAreaName ].Text = ((BloodBankAccessionArea)_accessionAreas.GetByObjectID( _division.AccessionAreaId )).Name;

				SyncLabelPrinterUsageColumns();
				SyncTimeZoneConfigColumns();	

				SetStatusForeColor( _division.IsActive );
			}

			private void SyncLabelPrinterUsageColumns()
			{
				foreach (string printer in Common.Utility.GetPrintersInstalledOnServer())
				{
					if (_division.PrinterName.Trim() == printer.Trim())
					{
						this.SubItems[ (int)DivSubItemIndex.PrinterName ].Text = StUiNullConvert.From( _division.PrinterName );
						break;
					}
				}

				if( _division.UsesLabelPrinter )
				{
					this.SubItems[ (int)DivSubItemIndex.LabelPrinterComPort ].Text = StUiNullConvert.From( _division.LabelPrinterConfig.ComPortNumber );
					this.SubItems[ (int)DivSubItemIndex.LabelPrinterIpAddress ].Text = StUiNullConvert.From( _division.LabelPrinterConfig.ListenerIPAddress );
					this.SubItems[ (int)DivSubItemIndex.LabelPrinterTcpPort ].Text = StUiNullConvert.From( _division.LabelPrinterConfig.ListenerTcpPortNumber );
				}
				else
				{
					this.SubItems[ (int)DivSubItemIndex.LabelPrinterComPort ].Text = 
						this.SubItems[ (int)DivSubItemIndex.LabelPrinterIpAddress ].Text = 
						this.SubItems[ (int)DivSubItemIndex.LabelPrinterTcpPort ].Text = NaString;
				}
			}

			private void SyncTimeZoneConfigColumns()
			{
				this.SubItems[ (int)DivSubItemIndex.TimeZone ].Text = _division.TimeZoneConfig.TimeZone.StandardTimeText;
				this.SubItems[ (int)DivSubItemIndex.DaylightSavingsType ].Text = ((DaylightSavingsTypeDefinition)_daylightSavingsTypes.GetByObjectID( _division.TimeZoneConfig.DaylightSavingsType )).Name;
				
				if( _division.TimeZoneConfig.DaylightSavingsType == DaylightSavingsType.Custom )
				{
					this.SubItems[ (int)DivSubItemIndex.CustomDstStartDate ].Text = _division.TimeZoneConfig.CustomDaylightSavingsStart.Month + "/" + _division.TimeZoneConfig.CustomDaylightSavingsStart.Day;
					this.SubItems[ (int)DivSubItemIndex.CustomDstEndDate ].Text = _division.TimeZoneConfig.CustomDaylightSavingsEnd.Month + "/" + _division.TimeZoneConfig.CustomDaylightSavingsEnd.Day;
				}
				else
				{
					this.SubItems[ (int)DivSubItemIndex.CustomDstStartDate ].Text = this.SubItems[ (int)DivSubItemIndex.CustomDstEndDate ].Text = NaString;
				}
			}

			/// <summary>
			/// GetListViewItemByDivision
			/// </summary>
			/// <param name="division"></param>
			/// <returns></returns>
			public static DivisionListViewItem GetListViewItemByDivision( DivisionBase division )
			{
				if( division == null )
					throw( new ArgumentNullException( "division" ) );

				lock( typeof( DivisionListViewItem ) )
					return (DivisionListViewItem)_allItems[ division ];
			}
		}

		#endregion

		#region 'No Data Found' ListViewItems implementations
		
		/// <summary>
		/// Class NoDivisionsDefinedListViewItem
		/// </summary>
		protected class NoDivisionsDefinedListViewItem : NoDataFoundListViewItem
		{
			/// <summary>
			/// Constructor
			/// </summary>
			public NoDivisionsDefinedListViewItem() 
				: base( StrRes.OtherMsg.MUC02.NoDivisionsDefinedInVbecs().ResString ) {}
		}

		/// <summary>
		/// Class ThereAreNoActiveDivisionsListViewItem
		/// </summary>
		protected class ThereAreNoActiveDivisionsListViewItem : NoDataFoundListViewItem
		{
			/// <summary>
			/// Constructor
			/// </summary>
			public ThereAreNoActiveDivisionsListViewItem() 
				: base( StrRes.OtherMsg.MUC02.NoActiveDivisionsToDisplay().ResString ) {}
		}

		#endregion

		private static BusinessObjectSortedHashList _daylightSavingsTypes, _accessionAreas, _nationalFacilities;

		private const string FORM_NAME = "Configure Division";

		private ArrayList _vbecsDivisionsSortedByCode, _vbecsDivisionsSortedByName;
		private ICollection _availableVistADivisions;

		private System.Windows.Forms.GroupBox grpDivisionConfiguration;
		private System.Windows.Forms.ListView lvDivisions;
		private System.Windows.Forms.CheckBox cbShowInactiveDivisions;
		private System.Windows.Forms.ColumnHeader colActive;
		private System.Windows.Forms.ColumnHeader colDivisionName;
		private System.Windows.Forms.ColumnHeader colDivisionCode;
		private System.Windows.Forms.ColumnHeader colFacility;
		private System.Windows.Forms.ColumnHeader colTimeout;
		private System.Windows.Forms.GroupBox grpIccbbaFacility;
		private System.Windows.Forms.Button btnFacilities;
		private System.Windows.Forms.ComboBox cmbFacilityName;
		private System.Windows.Forms.Label lblFacilityName;
		private System.Windows.Forms.GroupBox grpLockTimeout;
		private System.Windows.Forms.Label lblMins;
		private VbecsNumericUpDown updLockTimeout;
		private System.Windows.Forms.Label lblLockTimeout;
		private System.Windows.Forms.GroupBox grpDivisionStatus;
		private System.Windows.Forms.CheckBox cbActiveVBECSDivision;
		private System.Windows.Forms.GroupBox grpDivision;
		private System.Windows.Forms.ComboBox cmbDivisionCode;
		private System.Windows.Forms.Label lblDivisionCode;
		private System.Windows.Forms.ComboBox cmbDivisionName;
		private System.Windows.Forms.Label lblDivisionName;
		private System.Windows.Forms.Button btnSearchVistADivisons;
		private System.Windows.Forms.ColumnHeader colLabelPrinterComPortNumber;
		private System.Windows.Forms.ColumnHeader colLabelPrinterTcpPortNumber;
		private System.Windows.Forms.ColumnHeader colLabelPrinterIPAddress;
		private System.Windows.Forms.ColumnHeader colTimeZone;
		private System.Windows.Forms.ColumnHeader colDstType;
		private System.Windows.Forms.ColumnHeader colDstStartDate;
		private System.Windows.Forms.ColumnHeader colDstEndDate;
		private gov.va.med.vbecs.admin.GUI.EditDivisionLabelPrinterConfig editPrinterConfig;
		private System.Windows.Forms.ComboBox cmbAccessionArea;
		private System.Windows.Forms.GroupBox grpAccessionArea;
		private System.Windows.Forms.Label label5;
		private System.Windows.Forms.ColumnHeader colAccessionArea;
		private gov.va.med.vbecs.admin.GUI.EditDivisionTimeZoneConfig editTimeZoneConfig;
		private System.Windows.Forms.TabControl tabDivision;
		private System.Windows.Forms.TabPage pageDivisionIdentification;
		private System.Windows.Forms.TabPage pageLabelPrinting;
		private System.Windows.Forms.TabPage pageTimeZone;
		private System.Windows.Forms.GroupBox grpDivisionIdentificationBorder;
		private System.Windows.Forms.ColumnHeader colServiceType;
		private System.Windows.Forms.TabPage tabServiceType;
		private System.Windows.Forms.GroupBox grpServiceTypeBorder;
		private System.Windows.Forms.RadioButton rbServiceTypeTransfusionOnly;
		private System.Windows.Forms.RadioButton rbServiceTypeFull;
		private System.Windows.Forms.Panel pnlTestingCantBeConfiguredMessage;
		private System.Windows.Forms.PictureBox picAlertIcon;
		private System.Windows.Forms.Label lblTestingCantBeConfiguredText;
		private bool _formInitialized;
        private System.Windows.Forms.ListView lvInstitutions;
        private System.Windows.Forms.CheckBox cbIncludesInstitutions;
        private System.Windows.Forms.ColumnHeader InstitutionDivCodeColumn;
        private System.Windows.Forms.ListView listView1;
        private System.Windows.Forms.ColumnHeader InstitutionNameColumn;
		private System.Data.DataTable _assocInstitutions;
		private System.ComponentModel.Container components = null;
		private bool _refreshing = false;
		private System.Windows.Forms.ColumnHeader ActiveColumn;
		private gov.va.med.vbecs.GUI.controls.VbecsButton btnDivisionSave;
		private gov.va.med.vbecs.GUI.controls.VbecsButton btnDivisionClear;
		private System.Windows.Forms.ColumnHeader colReportPrinter;
		IList _vbecsDivisions;

		/// <summary>
		/// Constructor
		/// </summary>
		public EditDivisionParameters() 
			: base( DivBrRules.FormClassName )
		{
			// This call is required by the Windows.Forms Form Designer.
			InitializeComponent();						

			SetControlsMinMaxValues();

#if !DESIGN
			SetStaticTooltips();
#endif
			btnDivisionSave.Text = "&Save";
			btnDivisionClear.Text = "&Clear";

			LogonUser.NotDefined = true;
			_availableVistADivisions = null;
			_formInitialized = false;
		}

		/// <summary>
		/// BindValidator
		/// </summary>
		protected override void BindValidator()
		{
			InputValidator.SetControlBindings( cmbFacilityName, DivBrRules.FacilityNameNotSet );
			InputValidator.SetControlBindings( updLockTimeout, DivBrRules.LockInactivityTimeoutNotSet, DivBrRules.LockInactivityTimeoutInvalid );
			InputValidator.SetControlBindings( cmbAccessionArea, DivBrRules.AccessionAreaNotSet );

			// Although this in on a different form, it makes since to have it here since it's behavior (when it's enabled/ disabled)
			// is similar to this form.
			InputValidator.SetControlBindings( this.editPrinterConfig.cmbReportPrinter, DivBrRules.DefaultPrinterNotSet);
		}

		private void SetControlsMinMaxValues()
		{
			GuiUtility.SetNumericUpdownMinMax( updLockTimeout, DivisionBase.MinLockInactivityTimeoutMin, DivisionBase.MaxLockInactivityTimeoutMin );
		}
	
		private void SetStaticTooltips()
		{						
			SetToolTip( cbShowInactiveDivisions, StrRes.Tooltips.MUC02.CheckToDisplayInactive() );
			SetToolTip( cmbDivisionCode, StrRes.Tooltips.MUC02.SelectDivisionCode() );
			SetToolTip( cmbDivisionName, StrRes.Tooltips.MUC02.SelectDivisionName() );
			SetToolTip( btnSearchVistADivisons, StrRes.Tooltips.MUC02.ClickToSearchDivision() );
			SetToolTip( cmbFacilityName, StrRes.Tooltips.MUC02.SelectFacility() );
			SetToolTip( btnFacilities, StrRes.Tooltips.MUC02.ClickToSearchFacility() );
			SetToolTip( cmbAccessionArea, StrRes.Tooltips.MUC02.SelectAccessionArea() );
			SetToolTip( updLockTimeout, StrRes.Tooltips.MUC02.EnterLockInactivityTimeout() );
		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
            System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(EditDivisionParameters));
			this.grpDivisionConfiguration = new System.Windows.Forms.GroupBox();
			this.cbShowInactiveDivisions = new System.Windows.Forms.CheckBox();
			this.lvDivisions = new System.Windows.Forms.ListView();
			this.colActive = new System.Windows.Forms.ColumnHeader();
			this.colDivisionCode = new System.Windows.Forms.ColumnHeader();
			this.colDivisionName = new System.Windows.Forms.ColumnHeader();
			this.colFacility = new System.Windows.Forms.ColumnHeader();
			this.colServiceType = new System.Windows.Forms.ColumnHeader();
			this.colAccessionArea = new System.Windows.Forms.ColumnHeader();
			this.colTimeout = new System.Windows.Forms.ColumnHeader();
			this.colLabelPrinterComPortNumber = new System.Windows.Forms.ColumnHeader();
			this.colLabelPrinterTcpPortNumber = new System.Windows.Forms.ColumnHeader();
			this.colLabelPrinterIPAddress = new System.Windows.Forms.ColumnHeader();
			this.colReportPrinter = new System.Windows.Forms.ColumnHeader();
			this.colTimeZone = new System.Windows.Forms.ColumnHeader();
			this.colDstType = new System.Windows.Forms.ColumnHeader();
			this.colDstStartDate = new System.Windows.Forms.ColumnHeader();
			this.colDstEndDate = new System.Windows.Forms.ColumnHeader();
			this.editTimeZoneConfig = new gov.va.med.vbecs.admin.GUI.EditDivisionTimeZoneConfig();
			this.grpAccessionArea = new System.Windows.Forms.GroupBox();
			this.label5 = new System.Windows.Forms.Label();
			this.cmbAccessionArea = new System.Windows.Forms.ComboBox();
			this.editPrinterConfig = new gov.va.med.vbecs.admin.GUI.EditDivisionLabelPrinterConfig();
			this.grpIccbbaFacility = new System.Windows.Forms.GroupBox();
			this.btnFacilities = new System.Windows.Forms.Button();
			this.cmbFacilityName = new System.Windows.Forms.ComboBox();
			this.lblFacilityName = new System.Windows.Forms.Label();
			this.grpLockTimeout = new System.Windows.Forms.GroupBox();
			this.lblMins = new System.Windows.Forms.Label();
			this.updLockTimeout = new VbecsNumericUpDown();
			this.lblLockTimeout = new System.Windows.Forms.Label();
			this.grpDivisionStatus = new System.Windows.Forms.GroupBox();
			this.cbActiveVBECSDivision = new System.Windows.Forms.CheckBox();
			this.grpDivision = new System.Windows.Forms.GroupBox();
			this.cmbDivisionCode = new System.Windows.Forms.ComboBox();
			this.lblDivisionCode = new System.Windows.Forms.Label();
			this.cmbDivisionName = new System.Windows.Forms.ComboBox();
			this.lblDivisionName = new System.Windows.Forms.Label();
			this.btnSearchVistADivisons = new System.Windows.Forms.Button();
			this.tabDivision = new System.Windows.Forms.TabControl();
			this.pageDivisionIdentification = new System.Windows.Forms.TabPage();
			this.grpDivisionIdentificationBorder = new System.Windows.Forms.GroupBox();
			this.cbIncludesInstitutions = new System.Windows.Forms.CheckBox();
			this.lvInstitutions = new System.Windows.Forms.ListView();
			this.ActiveColumn = new System.Windows.Forms.ColumnHeader();
			this.InstitutionDivCodeColumn = new System.Windows.Forms.ColumnHeader();
			this.InstitutionNameColumn = new System.Windows.Forms.ColumnHeader();
			this.tabServiceType = new System.Windows.Forms.TabPage();
			this.grpServiceTypeBorder = new System.Windows.Forms.GroupBox();
			this.pnlTestingCantBeConfiguredMessage = new System.Windows.Forms.Panel();
			this.lblTestingCantBeConfiguredText = new System.Windows.Forms.Label();
			this.picAlertIcon = new System.Windows.Forms.PictureBox();
			this.rbServiceTypeTransfusionOnly = new System.Windows.Forms.RadioButton();
			this.rbServiceTypeFull = new System.Windows.Forms.RadioButton();
			this.pageLabelPrinting = new System.Windows.Forms.TabPage();
			this.pageTimeZone = new System.Windows.Forms.TabPage();
			this.listView1 = new System.Windows.Forms.ListView();
			this.btnDivisionSave = new gov.va.med.vbecs.GUI.controls.VbecsButton();
			this.btnDivisionClear = new gov.va.med.vbecs.GUI.controls.VbecsButton();
			this.grpDivisionConfiguration.SuspendLayout();
			this.grpAccessionArea.SuspendLayout();
			this.grpIccbbaFacility.SuspendLayout();
			this.grpLockTimeout.SuspendLayout();
			((System.ComponentModel.ISupportInitialize)(this.updLockTimeout)).BeginInit();
			this.grpDivisionStatus.SuspendLayout();
			this.grpDivision.SuspendLayout();
			this.tabDivision.SuspendLayout();
			this.pageDivisionIdentification.SuspendLayout();
			this.grpDivisionIdentificationBorder.SuspendLayout();
			this.tabServiceType.SuspendLayout();
			this.grpServiceTypeBorder.SuspendLayout();
			this.pnlTestingCantBeConfiguredMessage.SuspendLayout();
			this.pageLabelPrinting.SuspendLayout();
			this.pageTimeZone.SuspendLayout();
			this.SuspendLayout();
			// 
			// grpDivisionConfiguration
			// 
			this.grpDivisionConfiguration.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.grpDivisionConfiguration.Controls.Add(this.cbShowInactiveDivisions);
			this.grpDivisionConfiguration.Controls.Add(this.lvDivisions);
			this.grpDivisionConfiguration.Location = new System.Drawing.Point(0, 316);
			this.grpDivisionConfiguration.Name = "grpDivisionConfiguration";
			this.grpDivisionConfiguration.Size = new System.Drawing.Size(584, 192);
			this.grpDivisionConfiguration.TabIndex = 10;
			this.grpDivisionConfiguration.TabStop = false;
			this.grpDivisionConfiguration.Text = "VBECS Division Configuration";
			// 
			// cbShowInactiveDivisions
			// 
			this.cbShowInactiveDivisions.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left)));
			this.cbShowInactiveDivisions.Location = new System.Drawing.Point(8, 172);
			this.cbShowInactiveDivisions.Name = "cbShowInactiveDivisions";
			this.cbShowInactiveDivisions.Size = new System.Drawing.Size(152, 16);
			this.cbShowInactiveDivisions.TabIndex = 13;
			this.cbShowInactiveDivisions.Text = "Show Inactive Divisions";
			this.cbShowInactiveDivisions.CheckedChanged += new System.EventHandler(this.cbShowInactiveDivisions_CheckedChanged);
			// 
			// lvDivisions
			// 
			this.lvDivisions.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.lvDivisions.CausesValidation = false;
			this.lvDivisions.CheckBoxes = true;
			this.lvDivisions.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
																						  this.colActive,
																						  this.colDivisionCode,
																						  this.colDivisionName,
																						  this.colFacility,
																						  this.colServiceType,
																						  this.colAccessionArea,
																						  this.colTimeout,
																						  this.colLabelPrinterComPortNumber,
																						  this.colLabelPrinterTcpPortNumber,
																						  this.colLabelPrinterIPAddress,
																						  this.colReportPrinter,
																						  this.colTimeZone,
																						  this.colDstType,
																						  this.colDstStartDate,
																						  this.colDstEndDate});
			this.lvDivisions.FullRowSelect = true;
			this.lvDivisions.HideSelection = false;
			this.lvDivisions.Location = new System.Drawing.Point(8, 15);
			this.lvDivisions.MultiSelect = false;
			this.lvDivisions.Name = "lvDivisions";
			this.lvDivisions.Size = new System.Drawing.Size(568, 153);
			this.lvDivisions.TabIndex = 12;
			this.lvDivisions.View = System.Windows.Forms.View.Details;
			this.lvDivisions.ColumnClick += new System.Windows.Forms.ColumnClickEventHandler(this.lvDivisions_ColumnClick);
			this.lvDivisions.SelectedIndexChanged += new System.EventHandler(this.lvDivisions_SelectedIndexChanged);
			this.lvDivisions.ItemCheck += new System.Windows.Forms.ItemCheckEventHandler(this.lvDivisions_ItemCheck);
			// 
			// colActive
			// 
			this.colActive.Text = "Active";
			this.colActive.Width = 45;
			// 
			// colDivisionCode
			// 
			this.colDivisionCode.Text = "Division Code";
			this.colDivisionCode.Width = 80;
			// 
			// colDivisionName
			// 
			this.colDivisionName.Text = "Division Name";
			this.colDivisionName.Width = 150;
			// 
			// colFacility
			// 
			this.colFacility.Text = "Facility Name";
			this.colFacility.Width = 100;
			// 
			// colServiceType
			// 
			this.colServiceType.Text = "Service Type";
			this.colServiceType.Width = 100;
			// 
			// colAccessionArea
			// 
			this.colAccessionArea.Text = "Accession Area";
			this.colAccessionArea.Width = 100;
			// 
			// colTimeout
			// 
			this.colTimeout.Text = "Lock Timeout";
			this.colTimeout.Width = 75;
			// 
			// colLabelPrinterComPortNumber
			// 
			this.colLabelPrinterComPortNumber.Text = "Printer COM Port";
			this.colLabelPrinterComPortNumber.Width = 85;
			// 
			// colLabelPrinterTcpPortNumber
			// 
			this.colLabelPrinterTcpPortNumber.Text = "Printer TCP Port";
			// 
			// colLabelPrinterIPAddress
			// 
			this.colLabelPrinterIPAddress.Text = "Printer IP Address";
			// 
			// colReportPrinter
			// 
			this.colReportPrinter.Text = "Report Printer";
			// 
			// colTimeZone
			// 
			this.colTimeZone.Text = "Time Zone";
			// 
			// colDstType
			// 
			this.colDstType.Text = "Daylight Savings";
			// 
			// colDstStartDate
			// 
			this.colDstStartDate.Text = "DST Start";
			// 
			// colDstEndDate
			// 
			this.colDstEndDate.Text = "DST End";
			// 
			// editTimeZoneConfig
			// 
			this.editTimeZoneConfig.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.editTimeZoneConfig.Location = new System.Drawing.Point(0, 0);
			this.editTimeZoneConfig.Name = "editTimeZoneConfig";
			this.editTimeZoneConfig.Size = new System.Drawing.Size(576, 258);
			this.editTimeZoneConfig.TabIndex = 27;
			this.editTimeZoneConfig.ControlDataChanged += new System.EventHandler(this.AggregatedControlsDataChanged);
			// 
			// grpAccessionArea
			// 
			this.grpAccessionArea.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.grpAccessionArea.Controls.Add(this.label5);
			this.grpAccessionArea.Controls.Add(this.cmbAccessionArea);
			this.grpAccessionArea.Location = new System.Drawing.Point(296, 56);
			this.grpAccessionArea.Name = "grpAccessionArea";
			this.grpAccessionArea.Size = new System.Drawing.Size(276, 44);
			this.grpAccessionArea.TabIndex = 26;
			this.grpAccessionArea.TabStop = false;
			this.grpAccessionArea.Text = "Accession Area";
			// 
			// label5
			// 
			this.label5.Location = new System.Drawing.Point(8, 16);
			this.label5.Name = "label5";
			this.label5.Size = new System.Drawing.Size(68, 20);
			this.label5.TabIndex = 34;
			this.label5.Text = "Area Name*";
			this.label5.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// cmbAccessionArea
			// 
			this.cmbAccessionArea.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.cmbAccessionArea.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.erpMainProvider.SetIconAlignment(this.cmbAccessionArea, System.Windows.Forms.ErrorIconAlignment.MiddleLeft);
			this.cmbAccessionArea.Location = new System.Drawing.Point(84, 16);
			this.cmbAccessionArea.Name = "cmbAccessionArea";
			this.cmbAccessionArea.Size = new System.Drawing.Size(184, 21);
			this.cmbAccessionArea.TabIndex = 7;
			this.cmbAccessionArea.SelectedIndexChanged += new System.EventHandler(this.DataEntryFields_DataChanged);
			// 
			// editPrinterConfig
			// 
			this.editPrinterConfig.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.editPrinterConfig.Location = new System.Drawing.Point(0, 0);
			this.editPrinterConfig.Name = "editPrinterConfig";
			this.editPrinterConfig.Size = new System.Drawing.Size(576, 258);
			this.editPrinterConfig.TabIndex = 25;
			this.editPrinterConfig.ControlDataChanged += new System.EventHandler(this.AggregatedControlsDataChanged);
			// 
			// grpIccbbaFacility
			// 
			this.grpIccbbaFacility.Controls.Add(this.btnFacilities);
			this.grpIccbbaFacility.Controls.Add(this.cmbFacilityName);
			this.grpIccbbaFacility.Controls.Add(this.lblFacilityName);
			this.grpIccbbaFacility.Location = new System.Drawing.Point(296, 8);
			this.grpIccbbaFacility.Name = "grpIccbbaFacility";
			this.grpIccbbaFacility.Size = new System.Drawing.Size(276, 44);
			this.grpIccbbaFacility.TabIndex = 11;
			this.grpIccbbaFacility.TabStop = false;
			this.grpIccbbaFacility.Text = "Associated FDA Registered Facility";
			// 
			// btnFacilities
			// 
			this.btnFacilities.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.btnFacilities.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.btnFacilities.Location = new System.Drawing.Point(244, 16);
			this.btnFacilities.Name = "btnFacilities";
			this.btnFacilities.Size = new System.Drawing.Size(22, 22);
			this.btnFacilities.TabIndex = 6;
			this.btnFacilities.Text = "...";
			this.btnFacilities.Click += new System.EventHandler(this.btnFacilities_Click);
			// 
			// cmbFacilityName
			// 
			this.cmbFacilityName.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.cmbFacilityName.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.erpMainProvider.SetIconAlignment(this.cmbFacilityName, System.Windows.Forms.ErrorIconAlignment.MiddleLeft);
			this.cmbFacilityName.Location = new System.Drawing.Point(84, 16);
			this.cmbFacilityName.Name = "cmbFacilityName";
			this.cmbFacilityName.Size = new System.Drawing.Size(160, 21);
			this.cmbFacilityName.TabIndex = 5;
			this.cmbFacilityName.SelectedIndexChanged += new System.EventHandler(this.DataEntryFields_DataChanged);
			// 
			// lblFacilityName
			// 
			this.lblFacilityName.Location = new System.Drawing.Point(8, 16);
			this.lblFacilityName.Name = "lblFacilityName";
			this.lblFacilityName.Size = new System.Drawing.Size(84, 20);
			this.lblFacilityName.TabIndex = 33;
			this.lblFacilityName.Text = "Facility Name*";
			this.lblFacilityName.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// grpLockTimeout
			// 
			this.grpLockTimeout.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.grpLockTimeout.Controls.Add(this.lblMins);
			this.grpLockTimeout.Controls.Add(this.updLockTimeout);
			this.grpLockTimeout.Controls.Add(this.lblLockTimeout);
			this.grpLockTimeout.Location = new System.Drawing.Point(296, 104);
			this.grpLockTimeout.Name = "grpLockTimeout";
			this.grpLockTimeout.Size = new System.Drawing.Size(276, 40);
			this.grpLockTimeout.TabIndex = 10;
			this.grpLockTimeout.TabStop = false;
			this.grpLockTimeout.Text = "Lock Timeout";
			// 
			// lblMins
			// 
			this.lblMins.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.lblMins.Location = new System.Drawing.Point(244, 16);
			this.lblMins.Name = "lblMins";
			this.lblMins.Size = new System.Drawing.Size(28, 20);
			this.lblMins.TabIndex = 1;
			this.lblMins.Text = "mins";
			// 
			// updLockTimeout
			// 
			this.updLockTimeout.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.updLockTimeout.Location = new System.Drawing.Point(192, 14);
			this.updLockTimeout.Maximum = new System.Decimal(new int[] {
																		   15,
																		   0,
																		   0,
																		   0});
			this.updLockTimeout.Minimum = new System.Decimal(new int[] {
																		   5,
																		   0,
																		   0,
																		   0});
			this.updLockTimeout.Name = "updLockTimeout";
			this.updLockTimeout.Size = new System.Drawing.Size(52, 20);
			this.updLockTimeout.TabIndex = 8;
			this.updLockTimeout.Value = new System.Decimal(new int[] {
																		 5,
																		 0,
																		 0,
																		 0});
			this.updLockTimeout.KeyUp += new System.Windows.Forms.KeyEventHandler(this.NumericUpDown_KeyUp);
			this.updLockTimeout.ValueChanged += new System.EventHandler(this.DataEntryFields_DataChanged);
			// 
			// lblLockTimeout
			// 
			this.lblLockTimeout.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.lblLockTimeout.Location = new System.Drawing.Point(8, 14);
			this.lblLockTimeout.Name = "lblLockTimeout";
			this.lblLockTimeout.Size = new System.Drawing.Size(200, 20);
			this.lblLockTimeout.TabIndex = 21;
			this.lblLockTimeout.Text = "Lock Inactivity Timeout*";
			this.lblLockTimeout.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// grpDivisionStatus
			// 
			this.grpDivisionStatus.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.grpDivisionStatus.Controls.Add(this.cbActiveVBECSDivision);
			this.grpDivisionStatus.Location = new System.Drawing.Point(296, 148);
			this.grpDivisionStatus.Name = "grpDivisionStatus";
			this.grpDivisionStatus.Size = new System.Drawing.Size(276, 36);
			this.grpDivisionStatus.TabIndex = 9;
			this.grpDivisionStatus.TabStop = false;
			this.grpDivisionStatus.Text = "Status";
			// 
			// cbActiveVBECSDivision
			// 
			this.cbActiveVBECSDivision.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.cbActiveVBECSDivision.CheckAlign = System.Drawing.ContentAlignment.MiddleRight;
			this.cbActiveVBECSDivision.Location = new System.Drawing.Point(8, 12);
			this.cbActiveVBECSDivision.Name = "cbActiveVBECSDivision";
			this.cbActiveVBECSDivision.Size = new System.Drawing.Size(260, 18);
			this.cbActiveVBECSDivision.TabIndex = 9;
			this.cbActiveVBECSDivision.Text = "Active VBECS Division?";
			this.cbActiveVBECSDivision.TextAlign = System.Drawing.ContentAlignment.MiddleRight;
			this.cbActiveVBECSDivision.CheckedChanged += new System.EventHandler(this.cbActiveVBECSDivision_CheckedChanged);
			// 
			// grpDivision
			// 
			this.grpDivision.Controls.Add(this.cmbDivisionCode);
			this.grpDivision.Controls.Add(this.lblDivisionCode);
			this.grpDivision.Controls.Add(this.cmbDivisionName);
			this.grpDivision.Controls.Add(this.lblDivisionName);
			this.grpDivision.Controls.Add(this.btnSearchVistADivisons);
			this.grpDivision.Location = new System.Drawing.Point(4, 8);
			this.grpDivision.Name = "grpDivision";
			this.grpDivision.Size = new System.Drawing.Size(288, 76);
			this.grpDivision.TabIndex = 5;
			this.grpDivision.TabStop = false;
			this.grpDivision.Text = "Division";
			// 
			// cmbDivisionCode
			// 
			this.cmbDivisionCode.DisplayMember = "DivisionCode";
			this.cmbDivisionCode.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.cmbDivisionCode.Location = new System.Drawing.Point(84, 16);
			this.cmbDivisionCode.Name = "cmbDivisionCode";
			this.cmbDivisionCode.Size = new System.Drawing.Size(84, 21);
			this.cmbDivisionCode.TabIndex = 0;
			this.cmbDivisionCode.ValueMember = "DivisionCode";
			this.cmbDivisionCode.SelectedIndexChanged += new System.EventHandler(this.cmbDivisionCode_SelectedIndexChanged);
			// 
			// lblDivisionCode
			// 
			this.lblDivisionCode.Location = new System.Drawing.Point(8, 16);
			this.lblDivisionCode.Name = "lblDivisionCode";
			this.lblDivisionCode.Size = new System.Drawing.Size(80, 20);
			this.lblDivisionCode.TabIndex = 29;
			this.lblDivisionCode.Text = "Division Code*";
			this.lblDivisionCode.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// cmbDivisionName
			// 
			this.cmbDivisionName.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.cmbDivisionName.DisplayMember = "DivisionName";
			this.cmbDivisionName.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.cmbDivisionName.Location = new System.Drawing.Point(84, 44);
			this.cmbDivisionName.Name = "cmbDivisionName";
			this.cmbDivisionName.Size = new System.Drawing.Size(196, 21);
			this.cmbDivisionName.TabIndex = 2;
			this.cmbDivisionName.ValueMember = "DivisionName";
			this.cmbDivisionName.SelectedIndexChanged += new System.EventHandler(this.cmbDivisionName_SelectedIndexChanged);
			// 
			// lblDivisionName
			// 
			this.lblDivisionName.Location = new System.Drawing.Point(8, 44);
			this.lblDivisionName.Name = "lblDivisionName";
			this.lblDivisionName.Size = new System.Drawing.Size(84, 20);
			this.lblDivisionName.TabIndex = 3;
			this.lblDivisionName.Text = "Division Name*";
			this.lblDivisionName.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// btnSearchVistADivisons
			// 
            this.btnSearchVistADivisons.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.btnSearchVistADivisons.Location = new System.Drawing.Point(168, 16);
			this.btnSearchVistADivisons.Name = "btnSearchVistADivisons";
			this.btnSearchVistADivisons.Size = new System.Drawing.Size(22, 22);
			this.btnSearchVistADivisons.TabIndex = 1;
			this.btnSearchVistADivisons.Text = "...";
			this.btnSearchVistADivisons.Click += new System.EventHandler(this.btnSearchVistADivisons_Click);
			// 
			// tabDivision
			// 
			this.tabDivision.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.tabDivision.Controls.Add(this.pageDivisionIdentification);
			this.tabDivision.Controls.Add(this.tabServiceType);
			this.tabDivision.Controls.Add(this.pageLabelPrinting);
			this.tabDivision.Controls.Add(this.pageTimeZone);
			this.tabDivision.Location = new System.Drawing.Point(0, 0);
			this.tabDivision.Name = "tabDivision";
			this.tabDivision.SelectedIndex = 0;
			this.tabDivision.Size = new System.Drawing.Size(584, 284);
			this.tabDivision.TabIndex = 11;
			// 
			// pageDivisionIdentification
			// 
			this.pageDivisionIdentification.Controls.Add(this.grpDivision);
			this.pageDivisionIdentification.Controls.Add(this.grpDivisionIdentificationBorder);
			this.pageDivisionIdentification.Location = new System.Drawing.Point(4, 22);
			this.pageDivisionIdentification.Name = "pageDivisionIdentification";
			this.pageDivisionIdentification.Size = new System.Drawing.Size(576, 258);
			this.pageDivisionIdentification.TabIndex = 2;
			this.pageDivisionIdentification.Text = "Division Identification";
			// 
			// grpDivisionIdentificationBorder
			// 
			this.grpDivisionIdentificationBorder.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.grpDivisionIdentificationBorder.Controls.Add(this.cbIncludesInstitutions);
			this.grpDivisionIdentificationBorder.Controls.Add(this.lvInstitutions);
			this.grpDivisionIdentificationBorder.Controls.Add(this.grpLockTimeout);
			this.grpDivisionIdentificationBorder.Controls.Add(this.grpDivisionStatus);
			this.grpDivisionIdentificationBorder.Controls.Add(this.grpIccbbaFacility);
			this.grpDivisionIdentificationBorder.Controls.Add(this.grpAccessionArea);
			this.grpDivisionIdentificationBorder.Location = new System.Drawing.Point(0, 0);
			this.grpDivisionIdentificationBorder.Name = "grpDivisionIdentificationBorder";
			this.grpDivisionIdentificationBorder.Size = new System.Drawing.Size(576, 258);
			this.grpDivisionIdentificationBorder.TabIndex = 27;
			this.grpDivisionIdentificationBorder.TabStop = false;
			// 
			// cbIncludesInstitutions
			// 
			this.cbIncludesInstitutions.Enabled = false;
			this.cbIncludesInstitutions.Location = new System.Drawing.Point(8, 88);
			this.cbIncludesInstitutions.Name = "cbIncludesInstitutions";
			this.cbIncludesInstitutions.Size = new System.Drawing.Size(244, 16);
			this.cbIncludesInstitutions.TabIndex = 3;
			this.cbIncludesInstitutions.Text = "Map orders from VistA institutions";
			this.cbIncludesInstitutions.CheckedChanged += new System.EventHandler(this.cbIncludesInstitutions_CheckedChanged);
			// 
			// lvInstitutions
			// 
			this.lvInstitutions.BackColor = System.Drawing.SystemColors.Control;
			this.lvInstitutions.CheckBoxes = true;
			this.lvInstitutions.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
																							 this.ActiveColumn,
																							 this.InstitutionDivCodeColumn,
																							 this.InstitutionNameColumn});
			this.lvInstitutions.Location = new System.Drawing.Point(4, 108);
			this.lvInstitutions.Name = "lvInstitutions";
			this.lvInstitutions.Size = new System.Drawing.Size(288, 148);
			this.lvInstitutions.TabIndex = 4;
			this.lvInstitutions.View = System.Windows.Forms.View.Details;
			this.lvInstitutions.ColumnClick += new System.Windows.Forms.ColumnClickEventHandler(this.lvInstitutions_ColumnClick);
			this.lvInstitutions.ItemCheck += new System.Windows.Forms.ItemCheckEventHandler(this.lvInstitutions_ItemCheck);
			// 
			// ActiveColumn
			// 
			this.ActiveColumn.Text = "Active";
			this.ActiveColumn.Width = 45;
			// 
			// InstitutionDivCodeColumn
			// 
			this.InstitutionDivCodeColumn.Text = "Division Code";
			this.InstitutionDivCodeColumn.Width = 113;
			// 
			// InstitutionNameColumn
			// 
			this.InstitutionNameColumn.Text = "Institution Name";
			this.InstitutionNameColumn.Width = 160;
			// 
			// tabServiceType
			// 
			this.tabServiceType.Controls.Add(this.grpServiceTypeBorder);
			this.tabServiceType.Location = new System.Drawing.Point(4, 22);
			this.tabServiceType.Name = "tabServiceType";
			this.tabServiceType.Size = new System.Drawing.Size(576, 258);
			this.tabServiceType.TabIndex = 5;
			this.tabServiceType.Text = "Service Type";
			// 
			// grpServiceTypeBorder
			// 
			this.grpServiceTypeBorder.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.grpServiceTypeBorder.Controls.Add(this.pnlTestingCantBeConfiguredMessage);
			this.grpServiceTypeBorder.Controls.Add(this.rbServiceTypeTransfusionOnly);
			this.grpServiceTypeBorder.Controls.Add(this.rbServiceTypeFull);
			this.grpServiceTypeBorder.Location = new System.Drawing.Point(0, 0);
			this.grpServiceTypeBorder.Name = "grpServiceTypeBorder";
			this.grpServiceTypeBorder.Size = new System.Drawing.Size(576, 258);
			this.grpServiceTypeBorder.TabIndex = 0;
			this.grpServiceTypeBorder.TabStop = false;
			// 
			// pnlTestingCantBeConfiguredMessage
			// 
			this.pnlTestingCantBeConfiguredMessage.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.pnlTestingCantBeConfiguredMessage.Controls.Add(this.lblTestingCantBeConfiguredText);
			this.pnlTestingCantBeConfiguredMessage.Controls.Add(this.picAlertIcon);
			this.pnlTestingCantBeConfiguredMessage.Location = new System.Drawing.Point(8, 64);
			this.pnlTestingCantBeConfiguredMessage.Name = "pnlTestingCantBeConfiguredMessage";
			this.pnlTestingCantBeConfiguredMessage.Size = new System.Drawing.Size(560, 64);
			this.pnlTestingCantBeConfiguredMessage.TabIndex = 4;
			// 
			// lblTestingCantBeConfiguredText
			// 
			this.lblTestingCantBeConfiguredText.Location = new System.Drawing.Point(32, 8);
			this.lblTestingCantBeConfiguredText.Name = "lblTestingCantBeConfiguredText";
			this.lblTestingCantBeConfiguredText.Size = new System.Drawing.Size(520, 48);
			this.lblTestingCantBeConfiguredText.TabIndex = 1;
			this.lblTestingCantBeConfiguredText.Text = "There are open or partially completed worksheets or processes for DIVISION. Testi" +
				"ng cannot be configured until all work is completed.";
			// 
			// picAlertIcon
			// 
			this.picAlertIcon.Image = ((System.Drawing.Image)(resources.GetObject("picAlertIcon.Image")));
			this.picAlertIcon.Location = new System.Drawing.Point(8, 8);
			this.picAlertIcon.Name = "picAlertIcon";
			this.picAlertIcon.Size = new System.Drawing.Size(16, 16);
			this.picAlertIcon.TabIndex = 0;
			this.picAlertIcon.TabStop = false;
			// 
			// rbServiceTypeTransfusionOnly
			// 
			this.rbServiceTypeTransfusionOnly.Location = new System.Drawing.Point(16, 40);
			this.rbServiceTypeTransfusionOnly.Name = "rbServiceTypeTransfusionOnly";
			this.rbServiceTypeTransfusionOnly.Size = new System.Drawing.Size(152, 16);
			this.rbServiceTypeTransfusionOnly.TabIndex = 3;
			this.rbServiceTypeTransfusionOnly.Text = "Transfusion-Only Facility";
			this.rbServiceTypeTransfusionOnly.CheckedChanged += new System.EventHandler(this.DataEntryFields_DataChanged);
			// 
			// rbServiceTypeFull
			// 
			this.rbServiceTypeFull.Location = new System.Drawing.Point(16, 16);
			this.rbServiceTypeFull.Name = "rbServiceTypeFull";
			this.rbServiceTypeFull.Size = new System.Drawing.Size(128, 16);
			this.rbServiceTypeFull.TabIndex = 2;
			this.rbServiceTypeFull.Text = "Full-Service Facility";
			this.rbServiceTypeFull.CheckedChanged += new System.EventHandler(this.DataEntryFields_DataChanged);
			// 
			// pageLabelPrinting
			// 
			this.pageLabelPrinting.Controls.Add(this.editPrinterConfig);
			this.pageLabelPrinting.Location = new System.Drawing.Point(4, 22);
			this.pageLabelPrinting.Name = "pageLabelPrinting";
			this.pageLabelPrinting.Size = new System.Drawing.Size(576, 258);
			this.pageLabelPrinting.TabIndex = 3;
			this.pageLabelPrinting.Text = "Printers";
			// 
			// pageTimeZone
			// 
			this.pageTimeZone.Controls.Add(this.editTimeZoneConfig);
			this.pageTimeZone.Location = new System.Drawing.Point(4, 22);
			this.pageTimeZone.Name = "pageTimeZone";
			this.pageTimeZone.Size = new System.Drawing.Size(576, 258);
			this.pageTimeZone.TabIndex = 4;
			this.pageTimeZone.Text = "Time Zone";
			// 
			// listView1
			// 
			this.listView1.CheckBoxes = true;
			this.listView1.Location = new System.Drawing.Point(4, 108);
			this.listView1.Name = "listView1";
			this.listView1.Size = new System.Drawing.Size(288, 76);
			this.listView1.TabIndex = 27;
			// 
			// btnDivisionSave
			// 
			this.btnDivisionSave.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.btnDivisionSave.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.btnDivisionSave.Image = null;
			this.btnDivisionSave.Location = new System.Drawing.Point(512, 292);
			this.btnDivisionSave.Name = "btnDivisionSave";
			this.btnDivisionSave.Size = new System.Drawing.Size(64, 20);
			this.btnDivisionSave.TabIndex = 11;
			this.btnDivisionSave.Click += new System.EventHandler(this.btnDivisionSave_Click);
			// 
			// btnDivisionClear
			// 
			this.btnDivisionClear.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.btnDivisionClear.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.btnDivisionClear.Image = null;
			this.btnDivisionClear.Location = new System.Drawing.Point(440, 292);
			this.btnDivisionClear.Name = "btnDivisionClear";
			this.btnDivisionClear.Size = new System.Drawing.Size(64, 20);
			this.btnDivisionClear.TabIndex = 10;
			this.btnDivisionClear.Click += new System.EventHandler(this.btnDivisionClear_Click);
			// 
			// EditDivisionParameters
			// 
			this.Controls.Add(this.btnDivisionClear);
			this.Controls.Add(this.btnDivisionSave);
			this.Controls.Add(this.tabDivision);
			this.Controls.Add(this.grpDivisionConfiguration);
           
           
            
			this.Name = "EditDivisionParameters";
			this.Size = new System.Drawing.Size(584, 508);
			this.VisibleChanged += new System.EventHandler(this.EditDivisionParameters_VisibleChanged);
			this.grpDivisionConfiguration.ResumeLayout(false);
			this.grpAccessionArea.ResumeLayout(false);
			this.grpIccbbaFacility.ResumeLayout(false);
			this.grpLockTimeout.ResumeLayout(false);
			((System.ComponentModel.ISupportInitialize)(this.updLockTimeout)).EndInit();
			this.grpDivisionStatus.ResumeLayout(false);
			this.grpDivision.ResumeLayout(false);
			this.tabDivision.ResumeLayout(false);
			this.pageDivisionIdentification.ResumeLayout(false);
			this.grpDivisionIdentificationBorder.ResumeLayout(false);
			this.tabServiceType.ResumeLayout(false);
			this.grpServiceTypeBorder.ResumeLayout(false);
			this.pnlTestingCantBeConfiguredMessage.ResumeLayout(false);
			this.pageLabelPrinting.ResumeLayout(false);
			this.pageTimeZone.ResumeLayout(false);
			this.ResumeLayout(false);

		}
		#endregion

		private void btnSearchVistADivisons_Click(object sender, System.EventArgs e)
		{			
			DivisionBase _div = PerformSearchGetDivisionByAddingOrFinding();
			
			if( _div == null ) // user cancelled search
				return; 

			if( !RequestClearCurrentSelection() ) // current selection contains changes and user declined to lose them
				return;

			// We need to make sure inactive division will be shown
			if( !_div.IsActive && !cbShowInactiveDivisions.Checked )
				cbShowInactiveDivisions.Checked = true;
			
			if (!cmbDivisionCode.Items.Contains (_div))
			{
				cmbDivisionCode.Items.Add( _div );
				cmbDivisionName.Items.Add( _div );
			}

			cmbDivisionCode.SelectedItem = _div;			
		}

		private void LoadAvailableDivisionsFromVistA()
		{
			if( _availableVistADivisions != null )
				return;

			if( !VistALink.EnsureAvailability() )
				return;
			
			using( new WaitCursor() )
				_availableVistADivisions = DivisionDefinition.GetDivisionDefListFromVistA();
		}

		private DivisionBase PerformSearchGetDivisionByAddingOrFinding()
		{
			LoadAvailableDivisionsFromVistA();

			dlgVistADivisions _dlg = new dlgVistADivisions( _availableVistADivisions, cmbDivisionCode.SelectedItem != null ? ((DivisionBase)cmbDivisionCode.SelectedItem).DivisionCode : null );

			if( ResultAfterHostRefresh( _dlg.ShowDialog( this ) != DialogResult.OK ) )
				return null;

			DivisionBase _div = FindVbecsDivision( _dlg.SelectedVistADivision );

			return _div != null ? _div : AddNewDivisionFromVistA( _dlg.SelectedVistADivision );
		}

		private DivisionBase AddNewDivisionFromVistA( DivisionDefinition srcDivisionDef )
		{
			DivisionBase _division = new DivisionBase( srcDivisionDef );
			
			_division.IsActive = true; // we need to make it active otherwise it won't be always shown.
			
			if( !_vbecsDivisionsSortedByCode.Contains( _division ) ) 
				_vbecsDivisionsSortedByCode.Add( _division );
		
			if( !_vbecsDivisionsSortedByName.Contains( _division ) ) 
				_vbecsDivisionsSortedByName.Add( _division );

			SortDivisionLists();

			return _division;
		}

		private DivisionBase FindVbecsDivision( DivisionDefinition division )
		{
			if( division == null )
				throw( new ArgumentNullException( "division" ) );

			int _divIndex = _vbecsDivisionsSortedByCode.BinarySearch( division, new DivisionDefinitionCodeComparer() );

			return _divIndex < 0 ? null : (DivisionBase)_vbecsDivisionsSortedByCode[ _divIndex ];
		}

		/// <summary>
		/// Replaces OnLoad
		/// </summary>
		protected override void OnLoadInit()
		{		
#if !DESIGN
			this.Parent.Refresh();			

			base.OnLoadInit();

			_formInitialized = true;
#endif
		}

		private void LoadFormDataLists()
		{
			LoadDaylightSavingsTypes();

			LoadNationalFacilities();
			this.Refresh();

			if( !VistALink.EnsureAvailability() )
				return;

			LoadAccessionAreas();
			this.Refresh();

			LoadDivisionsFromVbecs();
			this.Refresh();
		}

		private void LoadDaylightSavingsTypes()
		{
			if( _daylightSavingsTypes != null )
				return;

			_daylightSavingsTypes = BOL.DaylightSavingsTypeDefinition.GetAllSortedByUse();
		}

		private void LoadAccessionAreas()
		{
			if( _accessionAreas != null )
				return;

			_accessionAreas = BloodBankAccessionArea.GetAllSortedByName();
		}

		private void LoadNationalFacilities()
		{
			if( _nationalFacilities != null )
				return;

			_nationalFacilities = FacilityDefinition.GetAllSortedByName();
		}

		private void LoadDivisionsFromVbecs()
		{
			_vbecsDivisions = DivisionBase.GetBaseInfoForAllVbecsDivisions();

			//Need to ensure that Division Name has not been changed in Vista
			//If it has, we must update the Division in Vbecs first
			if (SynchronizeDivisionNames(_vbecsDivisions))
			{
				//A division name has changed, so repopulate our division list
				_vbecsDivisions = DivisionBase.GetBaseInfoForAllVbecsDivisions();
			}

			_vbecsDivisionsSortedByCode = new ArrayList( _vbecsDivisions.Count );
			_vbecsDivisionsSortedByName = new ArrayList( _vbecsDivisions.Count );

			foreach( DivisionBase _div in _vbecsDivisions )
			{
				_vbecsDivisionsSortedByCode.Add( _div );
				_vbecsDivisionsSortedByName.Add( _div );
			}
			SortDivisionLists();
		}

		private bool SynchronizeDivisionNames(IList vbecsDivisions)
		{
			bool divisionNamesRequiredSync = false;

			if (_availableVistADivisions == null)
			{
				LoadAvailableDivisionsFromVistA();
			}

			foreach (DivisionBase vbecsDiv in vbecsDivisions)
			{
				foreach (DivisionDefinition vistaDiv in _availableVistADivisions)
				{
					if (vbecsDiv.DivisionCode == vistaDiv.DivisionCode)
					{
						if (vbecsDiv.DivisionName != vistaDiv.DivisionName)
						{
							//Name has been updated in Vista, need to update Vbecs
							vbecsDiv.DivisionName = vistaDiv.DivisionName;
							SaveDivision(vbecsDiv);
							divisionNamesRequiredSync = true;
							break;
						}
					}
				}
			}

			return divisionNamesRequiredSync;
		}

		private void SortDivisionLists()
		{
			_vbecsDivisionsSortedByCode.Sort( new DivisionDefinitionCodeComparer() );
			_vbecsDivisionsSortedByName.Sort( new DivisionDefinitionNameComparer() );
		}

		private void PopulateDivisionsListView()
		{
			lvDivisions.BeginUpdate();
			lvDivisions.Items.Clear();

			foreach( DivisionBase _division in _vbecsDivisionsSortedByCode )
			{
				if( !cbShowInactiveDivisions.Checked && !_division.IsActive )
					continue;
				
				lvDivisions.Items.Add( new DivisionListViewItem( _division ) );
			}

			AddNoDataFoundListViewItemIfNeeded();
			UpdateListViewLock();

			lvDivisions.EndUpdate();
		}

		private void AddNoDataFoundListViewItemIfNeeded()
		{
			if( lvDivisions.Items.Count != 0 )
				return;

			if( _vbecsDivisionsSortedByCode.Count != 0 && !cbShowInactiveDivisions.Checked && !((DivisionBase)_vbecsDivisionsSortedByCode[0]).IsNew )
				lvDivisions.Items.Add( new ThereAreNoActiveDivisionsListViewItem() );
			else
				lvDivisions.Items.Add( new NoDivisionsDefinedListViewItem() );
		}

		private void PopulateOnFormListControls()
		{
			DataLoadMode.Lock();

			PopulateNationalFacilityComboBox();
			PopulateAccessionAreaComboBox();
			PopulateDivisionsListView();
			PopulateDivisionCodeNameComboBoxes();			

			DataLoadMode.Release();
		}

		private void PopulateAccessionAreaComboBox()
		{
			GuiUtility.PopulateComboBoxFromList( cmbAccessionArea, _accessionAreas, "Name" );
		}

		private void PopulateNationalFacilityComboBox()
		{
			GuiUtility.PopulateComboBoxFromList( cmbFacilityName, _nationalFacilities, "CombinedName" );
		}

		private void PopulateDivisionCodeNameComboBoxes()
		{			
			cmbDivisionCode.Items.Clear();
			cmbDivisionName.Items.Clear();

			foreach( DivisionBase _division in _vbecsDivisionsSortedByCode )
				if( cbShowInactiveDivisions.Checked || _division.IsActive )
					cmbDivisionCode.Items.Add( _division );				

			foreach( DivisionBase _division in _vbecsDivisionsSortedByName )
				if( cbShowInactiveDivisions.Checked || _division.IsActive )
					cmbDivisionName.Items.Add( _division );
		}
		//CR 2316, DR 2723 
		private void PopulateInstitutions()
		{
			_refreshing = true;
			this.lvInstitutions.Items.Clear();
			string sBoundDivisionCode = this.BoundDivision.DivisionCode;
			if( _availableVistADivisions == null ) // user cancelled search
				return; 
			//A list of all institutions bound to the selected divition
			 _assocInstitutions =  this.BoundDivision.AssociatedInstitutions;
			//A list of all institutions in the VistA database
			System.Data.DataTable dtavailableVistADivisions = DivisionDefinition.GetDivisionDefTableFromVistA();
			//A list of all institutions bound to any division.
			System.Data.DataTable dtAllAssocIntitutions = DAL.Division.GetVbecsAssociatedInstitution();
			//CR 2341 refreshing VAMC division list.
			_vbecsDivisions = DivisionBase.GetBaseInfoForAllVbecsDivisions();
			//Remove any rows returned from VistA that are already defined as VBECS divisions
			//loop through the vbecs divisions
			foreach( DivisionBase _div in _vbecsDivisions )
			{
				//Get a list of all vistadivisions matching the vbecs division and remove them, this should usually be one or zero rows
				System.Data.DataRow [] dtRows = dtavailableVistADivisions.Select(DivTable.DivisionCode + " = '" + _div.DivisionCode.ToString().Trim() + "'");
				foreach(System.Data.DataRow drVistaRow in dtRows)
				{
					//Don't remove inactivated divisions, those should be selectable.
					if(_div.IsActive)
					{
						dtavailableVistADivisions.Rows.Remove(	drVistaRow );
					}
				}
			}
			//In case the division they selected has not been saved as a VAMC division yet remove it from the list 
			System.Data.DataRow [] dtNotSavedRow = dtavailableVistADivisions.Select(DivTable.DivisionCode + " = '" + this.cmbDivisionCode.Text.Trim() + "'");
			foreach(System.Data.DataRow drVistaRow in dtNotSavedRow)
			{
				dtavailableVistADivisions.Rows.Remove(	drVistaRow );
			}

			//Loop through the associated institutions and remove any institutions mapped to other VAMC division.
			//Array list of all institutions mapped ot other VAMC divisions
			//This datatable already only contains active.
			System.Data.DataRow [] dtRemovalRows = dtAllAssocIntitutions.Select(InstitutionTable.DivisionCode + "  <>  '" + this.cmbDivisionCode.Text.Trim() + "'");
			int test = dtRemovalRows.Length;
			foreach(System.Data.DataRow drFilter in dtRemovalRows)
			{
				string sInstitutionDiv = drFilter[InstitutionTable.InstitutionDivisionCode].ToString().Trim();
				//Just check if it is in the table we want to remove it from.
				System.Data.DataRow [] dtFilterRows = dtavailableVistADivisions.Select(DivTable.DivisionCode + " = '" + sInstitutionDiv + "'");
				if(dtFilterRows.Length != 0)
				{
					dtavailableVistADivisions.Rows.Remove(	dtFilterRows[0] );
				}
			}
			
			
			//Need to check the listview items that are already associated with the selected division
			foreach( System.Data.DataRow dtRow in dtavailableVistADivisions.Rows)
			{
				string [] subItems = new string [3];
				
				subItems[0] = string.Empty;
				subItems[1] = dtRow[DivTable.DivisionCode].ToString().Trim();
				subItems[2] = dtRow[DivTable.DivisionName].ToString().Trim();
				
				ListViewItem newItem = new ListViewItem(subItems);
				newItem.Tag = dtRow;
				//Need to check the listview items that are already associated with the selected division
				if(_assocInstitutions!=null)
						newItem.Checked =  (_assocInstitutions.Select(InstitutionTable.InstitutionDivisionCode + " = '" + dtRow[DivTable.DivisionCode].ToString() + "' AND " +  InstitutionTable.RecordStatusCode + "='A'").Length>0);
				this.lvInstitutions.Items.Add(newItem);				
			}
			_refreshing = false;
		}

		private void cbShowInactiveDivisions_CheckedChanged(object sender, System.EventArgs e)
		{
			if( DataLoadMode.IsLocked )
				return;

			using( new WaitCursor() )
			{
				DataLoadMode.Lock();

				ResetToNoDivision();
				PopulateDivisionCodeNameComboBoxes();
				PopulateDivisionsListView();
				
				DataLoadMode.Release();
			}
		}

		private void cmbDivisionCode_SelectedIndexChanged(object sender, System.EventArgs e)
		{	
			if( DataLoadMode.IsLocked || !ConfirmAbandonmentOfChangesForComboBoxesIfNeeded() )
				return;
			
			SelectDivision( (DivisionBase)cmbDivisionCode.SelectedItem );
			this.cbIncludesInstitutions.Enabled =  this.BoundDivision != null;
			if(this.BoundDivision != null)
				PopulateInstitutions();
		}

		private void cmbDivisionName_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			if( DataLoadMode.IsLocked || !ConfirmAbandonmentOfChangesForComboBoxesIfNeeded() )
				return;

			SelectDivision( (DivisionBase)cmbDivisionName.SelectedItem );
			this.cbIncludesInstitutions.Enabled =  this.BoundDivision != null;
			if(this.BoundDivision != null)
				PopulateInstitutions();
		}

		private void lvDivisions_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			if( DataLoadMode.IsLocked )
				return;

			DivisionListViewItem _selectedItem = GetSelectedListViewItem();

			SelectDivision( _selectedItem != null ? _selectedItem.Division : null );
			this.cbIncludesInstitutions.Enabled =  this.BoundDivision != null;
			if(this.BoundDivision != null)
				PopulateInstitutions();
		}

		/// <summary>
		/// SelectDivision
		/// </summary>
		/// <param name="division"></param>
		protected virtual void SelectDivision( DivisionBase division )
		{			
			DivisionBase _previouslySelectedDivision = BoundDivision;

			if( division != null )
			{
				BindToDivision( division );
				this.cbIncludesInstitutions.Enabled = true;
			}
			else
			{
				ResetToNoDivision();
				OnDataChanged( true );
			}

			UpdateAfterDivisionChange( _previouslySelectedDivision );
		}

		private void DisplaySelectDivisionInListView( DivisionBase division )
		{
			DivisionListViewItem _lvi = DivisionListViewItem.GetListViewItemByDivision( division );
				
			if( GetSelectedListViewItem() != _lvi )
				lvDivisions.SelectedItems.Clear();
			
			if( _lvi == null )
				return;
									
			_lvi.Selected = true;
			_lvi.EnsureVisible();
		}

		/// <summary>
		/// ConfirmAbandonmentOfChangesIfNeeded
		/// </summary>
		/// <returns></returns>
		protected virtual bool ConfirmAbandonmentOfChangesIfNeeded()
		{
			if( BoundDivision == null || !BoundDivision.IsDirty )
				return true;

			return ResultAfterHostRefresh( GuiMessenger.ShowMessageBox( StrRes.ConfMsg.MUC02.DivisionDataNotSaved( BoundDivision.DivisionName ) ) == DialogResult.Yes );
		}

		/// <summary>
		/// ConfirmAbandonmentOfChangesForComboBoxesIfNeeded
		/// </summary>
		/// <returns></returns>
		protected virtual bool ConfirmAbandonmentOfChangesForComboBoxesIfNeeded()
		{
			if( ConfirmAbandonmentOfChangesIfNeeded() )
				return true;

			DataLoadMode.Lock();
			cmbDivisionName.SelectedItem = BoundDivision;
			cmbDivisionCode.SelectedItem = BoundDivision;
			DataLoadMode.Release();

			return false;
		}

		/// <summary>
		/// RestoreDivisionObjectFromDb
		/// </summary>
		/// <param name="division"></param>
		protected virtual void RestoreDivisionObjectFromDb( DivisionBase division )
		{
			if( division == null || division.IsNew || !division.IsDirty )
				return;

			using( new WaitCursor() )
				division.RefreshFromDb();
		}

		/// <summary>
		/// DisplayData
		/// </summary>
		protected override void DisplayData()
		{
			cmbDivisionCode.SelectedItem = cmbDivisionName.SelectedItem = BoundDivision;
								
			DisplaySelectDivisionInListView( BoundDivision );

			grpDivisionStatus.Enabled = true;
			cbActiveVBECSDivision.Checked = BoundDivision.IsActive;

			grpLockTimeout.Enabled = true;
			updLockTimeout.Value = StUiNullConvert.DecimalFrom( BoundDivision.LockInactivityTimeoutMin, BOL.LockManager.DefaultLockInactivityTimeoutMin );

			editPrinterConfig.BindToDivision( BoundDivision );
			editTimeZoneConfig.BindToDivision( BoundDivision );

			cmbFacilityName.SelectedItem = _nationalFacilities.GetByObjectID( BoundDivision.FacilityGuid );
			grpIccbbaFacility.Enabled = true;

			cmbAccessionArea.SelectedItem = _accessionAreas.GetByObjectID( BoundDivision.AccessionAreaId );
			grpAccessionArea.Enabled = true;
			
			DisplayServiceTypeInfo();
			
			this.editPrinterConfig.cmbReportPrinter.Enabled = true;

			// This will only load a printer that is in the combo box.  In other words, a printer that has been
			// deleted from the server won't load.
			bool printerFound = false;
			foreach (string printer in Common.Utility.GetPrintersInstalledOnServer())
			{
				
				if (BoundDivision.PrinterName.Trim() == printer.Trim())
				{
					this.editPrinterConfig.cmbReportPrinter.SelectedItem = BoundDivision.PrinterName;
					printerFound = true;
				}
			}
			if (!printerFound) this.editPrinterConfig.cmbReportPrinter.SelectedItem = null;

			// Do this to trigger the broken rule in the above case!
			if (this.editPrinterConfig.cmbReportPrinter.SelectedItem == null) BoundDivision.PrinterName = string.Empty;


			if (Common.Utility.GetPrintersInstalledOnServer().GetLength(0) == 1)
			{
				this.editPrinterConfig.cmbReportPrinter.SelectedItem = BoundDivision.PrinterName = (string) this.editPrinterConfig.cmbReportPrinter.Items[0];
			}

			btnDivisionClear.EnabledButton = true;
		}

		/// <summary>
		/// DisplayServiceTypeInfo
		/// </summary>
		protected void DisplayServiceTypeInfo()
		{
			grpServiceTypeBorder.Enabled = rbServiceTypeFull.Enabled = rbServiceTypeTransfusionOnly.Enabled = true;

			rbServiceTypeFull.Checked = BoundDivision.ServiceTypeIndicator;
			rbServiceTypeTransfusionOnly.Checked = !BoundDivision.ServiceTypeIndicator;

			using( new WaitCursor() )
				if( !BoundDivision.CachedIsBloodUnitNotInFinalStatus() )
					return;
			
			rbServiceTypeFull.Enabled = rbServiceTypeTransfusionOnly.Enabled = false;
			lblTestingCantBeConfiguredText.Text	= StrRes.InfoMsg.UC055.CannotConfigureServiceType( BoundDivision.DivisionName ).ResString;
			pnlTestingCantBeConfiguredMessage.Visible = true;
		}

		/// <summary>
		/// ResetUiToNoSelection
		/// </summary>
		protected override void ResetUiToNoSelection()
		{
			InputValidator.ClearValidationErrorsForAllBoundControls();

			ClearDivisionParamsControlGroup();
			DisableDivisionParamsControlGroup();			
			
			editPrinterConfig.ResetToNoDivision();
			editTimeZoneConfig.ResetToNoDivision();

			UpdateButtons();
		}

		private void ClearDivisionParamsControlGroup()
		{			
			DataLoadMode.Lock();

			cmbDivisionCode.SelectedItem = null;
			cmbDivisionName.SelectedItem = null;
			lvDivisions.SelectedItems.Clear();

			rbServiceTypeFull.Checked = rbServiceTypeTransfusionOnly.Checked = false;
			pnlTestingCantBeConfiguredMessage.Visible = false;

			cmbFacilityName.SelectedItem = null;
			cmbAccessionArea.SelectedItem = null;
			updLockTimeout.Value = BOL.LockManager.DefaultLockInactivityTimeoutMin;
			
			cbActiveVBECSDivision.Checked = false;

			DataLoadMode.Release();
		}

		private void DisableDivisionParamsControlGroup()
		{			
			grpIccbbaFacility.Enabled = 
			grpServiceTypeBorder.Enabled = 
			grpDivisionStatus.Enabled = 
			grpLockTimeout.Enabled = 
			grpAccessionArea.Enabled = false;
		}

		private void lvDivisions_ColumnClick(object sender, System.Windows.Forms.ColumnClickEventArgs e)
		{
			ColumnClickSorter.ColumnClickHandler( sender, e, true, ColumnClickSorter.ColumnSortType.String );
		}

		private void btnFacilities_Click(object sender, System.EventArgs e)
		{
			if( !VistALink.EnsureAvailability() )
				return;

			DlgFacilitySearch _dlg = new DlgFacilitySearch();
			
			if( ResultAfterHostRefresh( _dlg.ShowDialog( this ) == DialogResult.OK ) )
				cmbFacilityName.SelectedItem = _nationalFacilities.GetByObjectID( _dlg.SelectedFacilityGuid );
		}

		private void lvDivisions_ItemCheck(object sender, System.Windows.Forms.ItemCheckEventArgs e)
		{
			if( DataLoadMode.IsLocked )
				return;

			DivisionListViewItem _lvi = (DivisionListViewItem)lvDivisions.Items[e.Index];
			string sDivCode = string.Empty;

			sDivCode = _lvi.Division.DivisionCode;

		    //If they are activitating an inactive VAMC division

			if(e.NewValue == CheckState.Checked)
			{
				//Get all active associated institutions then check if the VAMC div they are trying to activate has been mapped to send 
				//orders to another VBECS division
				System.Data.DataTable dtAllAssocIntitutions = DAL.Division.GetVbecsAssociatedInstitution();
				System.Data.DataRow [] dtMatchedRows = dtAllAssocIntitutions.Select(InstitutionTable.InstitutionDivisionCode + "  =  '" + sDivCode + "'");
				if(dtMatchedRows.Length != 0)
				{
					GuiMessenger.ShowMessageBox( StrRes.InfoMsg.MUC03.CannotActivateDivision());
					e.NewValue = CheckState.Unchecked;
					_lvi.Checked = false;
					return;
				}
			}
				//if they are inactivating a division that has orders mapped.
			else if( e.NewValue == CheckState.Unchecked)
			{
				//Get all active associated institutions then check if the VAMC div they are trying to inactivate has orders mapped to it. 
				System.Data.DataTable dtAllAssocIntitutions = DAL.Division.GetVbecsAssociatedInstitution();
				System.Data.DataRow [] dtMatchedRows = dtAllAssocIntitutions.Select(InstitutionTable.DivisionCode + "  =  '" + sDivCode + "'");
				if(dtMatchedRows.Length != 0)
				{
					GuiMessenger.ShowMessageBox( StrRes.InfoMsg.MUC03.CannotInactivateDivision());
					e.NewValue = CheckState.Checked;
					_lvi.Checked = true;
					return;
				}
			}

			_lvi.Division.IsActive = ( e.NewValue == CheckState.Checked );

			if( cmbDivisionCode.SelectedItem == _lvi.Division )
			{
				e.NewValue = e.CurrentValue; // need to cancel it to correspond to the logic for selected division
				cbActiveVBECSDivision.Checked = _lvi.Division.IsActive;
				return; // we don't care about currently edited division.
			}			

			if( !ConfirmDivisionInactivationIfNeeded( _lvi.Division ) )
			{
				e.NewValue = CheckState.Checked;
				_lvi.Division.IsActive = true;
				_lvi.Division.IsDirty = false; // need to update it here since we've reverted the change.
				return;			
			}
			
			SaveDivision( _lvi.Division );

			DataLoadMode.Lock();
			_lvi.SyncWithDataObject();
			DataLoadMode.Release();
		}

		private void UpdateAfterDivisionChange( DivisionBase previouslySelectedDivision )
		{
			if( previouslySelectedDivision == null )
				return;

			DataLoadMode.Lock();

			if( !previouslySelectedDivision.IsNew )
			{
				RestoreDivisionObjectFromDb( previouslySelectedDivision );
				DivisionListViewItem.GetListViewItemByDivision( previouslySelectedDivision ).SyncWithDataObject();
			}
			else
			{
				_vbecsDivisionsSortedByCode.Remove( previouslySelectedDivision );
				_vbecsDivisionsSortedByName.Remove( previouslySelectedDivision );
								
				cmbDivisionCode.Items.Remove( previouslySelectedDivision );
				cmbDivisionName.Items.Remove( previouslySelectedDivision );				
			}

			DataLoadMode.Release();
		}

		/// <summary>
		/// TransferUserInputToBoundDivisionObject
		/// </summary>
		protected override void TransferUserInputToBoundDivisionObject()
		{
			FacilityDefinition _facility = (FacilityDefinition)cmbFacilityName.SelectedItem;
			BoundDivision.FacilityGuid = ( _facility == null ) ? StNullConvert.GuidNull : _facility.FacilityGuid;

			BloodBankAccessionArea _accessionArea = (BloodBankAccessionArea)cmbAccessionArea.SelectedItem;
			BoundDivision.AccessionAreaId = ( _accessionArea == null ) ? StNullConvert.Int32Null : _accessionArea.ID;

			BoundDivision.ServiceTypeIndicator = rbServiceTypeFull.Checked;

            try
            {   //CR3321 - updated to use VbecsNumericUpDown which throws a FormatException on .Get() if bad data
                BoundDivision.LockInactivityTimeoutMin = (int)updLockTimeout.Value;
            }
            catch(FormatException)
            {
                BoundDivision.IsDirty = true;   //set this since they made an edit anyway, meh, but Stas's code sucks
            }
			BoundDivision.IsActive = cbActiveVBECSDivision.Checked; 
		}

		/// <summary>
		/// DataChanged Event
		/// </summary>
		/// <param name="dontTransferInput"></param>
		protected override void OnDataChanged( bool dontTransferInput )
		{
			base.OnDataChanged( dontTransferInput );

			UpdateButtons();
			UpdateListViewLock();
		}

		private void DisableButton( VbecsButton button )
		{
			if( button == null )
				throw( new ArgumentNullException( "button" ) );

			button.EnabledButton = false;
			SetToolTip( button, String.Empty );
		}

		private void UpdateButtons()
		{
			UpdateClearButtonState();
			UpdateSaveButtonState();
		}

		private void UpdateClearButtonState()
		{
			if( BoundDivision == null )
			{
				return;
			}

			btnDivisionClear.EnabledButton = true;
			SetToolTip( btnDivisionClear, StrRes.Tooltips.MUC02.ClickToCancel() );
		}

		private void UpdateSaveButtonState()
		{
			if( BoundDivision == null )
			{
				DisableButton( btnDivisionSave );
				return;
			}

			btnDivisionSave.EnabledButton = BoundDivision.IsDirty && BoundDivision.IsValid;

			if( BoundDivision.IsValid && !BoundDivision.IsDirty )
				SetToolTip( btnDivisionSave, StrRes.Tooltips.Common.NoChangesToAddOrUpdate() );
			else 
				if( BoundDivision.IsValid && BoundDivision.IsDirty )
				SetToolTip( btnDivisionSave, StrRes.Tooltips.MUC02.SaveDivisionData() );
			else
				SetToolTip( btnDivisionSave, BoundDivision.GetBrokenRulesMessage( DivBrRules.FormClassName ) );
		}

		/// <summary>
		/// Get BusinessObjectUnderValidation
		/// </summary>
		protected override BaseBusinessObject BusinessObjectUnderValidation
		{
			get
			{
				return BoundDivision;
			}
		}

		/// <summary>
		/// AggregatedControlsDataChanged
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected virtual void AggregatedControlsDataChanged( object sender, EventArgs e )
		{
			OnDataChanged( true );
		}

		private void btnDivisionSave_Click(object sender, EventArgs e)
		{
			// Ask user if ready to save data
			if( gov.va.med.vbecs.GUI.GuiMessenger.ShowMessageBox( this, Common.StrRes.ConfMsg.Common.OKToSave(FORM_NAME) ) != DialogResult.Yes	)
			{
				return;
			}

			if( !ConfirmDivisionInactivationIfNeeded( BoundDivision ) )
				return;

			bool _addingNewDivision = BoundDivision.IsNew;
			if(this.cbIncludesInstitutions.Checked)
			{
				BoundDivision.AssociatedInstitutions = this._assocInstitutions;
			}
			else
			{
				BoundDivision.AssociatedInstitutions = null;
			}
			SaveDivision( BoundDivision );		
			UpdateListViewLock();
			UpdateAfterDivisionSave( _addingNewDivision );
			SelectDivision( BoundDivision );
			this.cbIncludesInstitutions.Enabled =  this.BoundDivision != null;
			if(this.BoundDivision != null)
				PopulateInstitutions();
		}
		
		private void SaveDivision( DivisionBase divisionToSave )
		{			
			if( divisionToSave == null )
				throw( new ArgumentNullException( "divisionToSave" ) );

			try
			{
				LogonUser.LogonUserDivisionCode = divisionToSave.DivisionCode;

				using( new WaitCursor() )
					divisionToSave.PersistDivisionBase( UpdateFunction.MUC02_EditDivisionParameters );
			}
			finally
			{
				LogonUser.LogonUserDivisionCode = null;
			}
		}

		private void btnDivisionClear_Click(object sender, System.EventArgs e)
		{
			RequestClearCurrentSelection();
		}

		private bool RequestClearCurrentSelection()
		{
			if( !ConfirmAbandonmentOfChangesIfNeeded() )
				return false;

			DivisionBase _previouslySelectedDivision = BoundDivision;			
			
			ResetToNoDivision();
			//CR 2316, DR 2723 
			this.cbIncludesInstitutions.Checked = false;
			this.lvInstitutions.Items.Clear();
			UpdateListViewLock();
			UpdateAfterDivisionChange( _previouslySelectedDivision );

			return true;
		}

		private void UpdateAfterDivisionSave( bool newDivisionAdded )
		{
			DataLoadMode.Lock();

			if( newDivisionAdded )
			{
				EnsureThereAreNoListViewDataNotFoundItems();
				lvDivisions.Items.Add( new DivisionListViewItem( BoundDivision ) );
			}
			else
			{
				GetSelectedListViewItem().SyncWithDataObject();
			}

			DataLoadMode.Release();
		}

		private void EnsureThereAreNoListViewDataNotFoundItems()
		{
			if( lvDivisions.Items.Count == 1 && lvDivisions.Items[0] is NoDataFoundListViewItem )
				lvDivisions.Items.RemoveAt( 0 );
		}

		/// <summary>
		/// ConfirmDivisionInactivationIfNeeded
		/// </summary>
		/// <param name="division"></param>
		/// <returns></returns>
		protected virtual bool ConfirmDivisionInactivationIfNeeded( DivisionBase division )
		{
			if( division == null )
				throw( new ArgumentNullException( "division" ) );

			if( !division.IsInactivationPending ) 
				return true;

			return ResultAfterHostRefresh( GuiMessenger.ShowMessageBox( StrRes.ConfMsg.MUC02.InactivateDivision( division.DivisionName ) ) == DialogResult.Yes );
		}

		/// <summary>
		/// ConfirmExit
		/// </summary>
		/// <returns></returns>
		public override bool ConfirmExit()
		{
			return ConfirmAbandonmentOfChangesIfNeeded();
		}

		private void cbActiveVBECSDivision_CheckedChanged(object sender, EventArgs e)
		{
			
			if( lvDivisions.SelectedItems.Count > 0 ) // Need to reflect change for selected division
			{
				string sDivCode = string.Empty;
				if(this.cmbDivisionCode.Text.Trim() == string.Empty)
				{
					DivisionListViewItem _selectedItem = GetSelectedListViewItem();
					sDivCode = _selectedItem.Division.DivisionCode;
				}
				else
				{
					sDivCode = cmbDivisionCode.Text.Trim();

				}
				//If they are activitating an inactive VAMC division
				if(cbActiveVBECSDivision.Checked && this.BoundDivision.IsActive == false)
				{
					//Get all active associated institutions then check if the VAMC div they are trying to activate has been mapped to send 
					//orders to another VBECS division
					System.Data.DataTable dtAllAssocIntitutions = DAL.Division.GetVbecsAssociatedInstitution();
					System.Data.DataRow [] dtMatchedRows = dtAllAssocIntitutions.Select(InstitutionTable.InstitutionDivisionCode + "  =  '" + sDivCode + "'");
					if(dtMatchedRows.Length != 0)
					{
						GuiMessenger.ShowMessageBox( StrRes.InfoMsg.MUC03.CannotActivateDivision());
						cbActiveVBECSDivision.Checked = false;
						return;
					}
				}
					//if they are inactivating a division that has orders mapped.
				else if( cbActiveVBECSDivision.Checked == false && this.BoundDivision.IsActive == true)
				{
					//Get all active associated institutions then check if the VAMC div they are trying to inactivate has orders mapped to it. 
					System.Data.DataTable dtAllAssocIntitutions = DAL.Division.GetVbecsAssociatedInstitution();
					System.Data.DataRow [] dtMatchedRows = dtAllAssocIntitutions.Select(InstitutionTable.DivisionCode + "  =  '" + sDivCode + "'");
					if(dtMatchedRows.Length != 0)
					{
						GuiMessenger.ShowMessageBox( StrRes.InfoMsg.MUC03.CannotInactivateDivision());
						cbActiveVBECSDivision.Checked = true;
						return;
					}
				}

				DataLoadMode.Lock();
				GetSelectedListViewItem().Checked = cbActiveVBECSDivision.Checked; 
				DataLoadMode.Release();
			}

			DataEntryFields_DataChanged( sender, e );
		}

		private DivisionListViewItem GetSelectedListViewItem()
		{
			return lvDivisions.SelectedItems.Count == 1 ? (DivisionListViewItem)lvDivisions.SelectedItems[0] : null;
		}

		private void UpdateListViewLock()
		{
			bool _noDataFound = ( lvDivisions.Items.Count == 1 && lvDivisions.Items[0] is NoDataFoundListViewItem );
			bool _isDisabled = _noDataFound || ( BoundDivision != null && ( BoundDivision.IsDirty || BoundDivision.IsNew ) );
			
			lvDivisions.BackColor = _isDisabled ? SystemColors.Control : SystemColors.Window;					

			if( _noDataFound && lvDivisions.Items[0] is ThereAreNoActiveDivisionsListViewItem && !( BoundDivision != null && BoundDivision.IsNew ) ) 
			{
				// There're divisions defined, but there're none shown because they're all inactive.
				grpDivisionConfiguration.Enabled = true;
				lvDivisions.Enabled = false;
			}
			else
			{
				// Depends. There may be divisions shown, or there may be no divisions at all. 
				lvDivisions.Enabled = grpDivisionConfiguration.Enabled = !_isDisabled;
			}

			// Multicolumn checked list looks bad when displaying "no data found" message, hence adjusting the style if needed. 
			lvDivisions.CheckBoxes = !_noDataFound;
			lvDivisions.View = _noDataFound ? View.List : View.Details;
		}

		private void EditDivisionParameters_VisibleChanged(Object sender,EventArgs e)
		{
			if (_formInitialized)
			{
				_formInitialized = false;

				using( new WaitCursor() )
				{
					LoadFormDataLists();
					PopulateOnFormListControls();

					// Init child controls on separate pages.
					editPrinterConfig.DoOnLoadInit();
					editTimeZoneConfig.DoOnLoadInit();
				}
			}
		}

		private void cbIncludesInstitutions_CheckedChanged(object sender, System.EventArgs e)
		{
			if(this.cbIncludesInstitutions.Checked)
			{
				this.lvInstitutions.BackColor = System.Drawing.Color.White;
			}
			else
			{
				this.lvInstitutions.BackColor = System.Drawing.SystemColors.Control;
			}	
			if(this.cbIncludesInstitutions.Checked && this.lvInstitutions.Items.Count == 0)
			{
				this.PopulateInstitutions();
			}
		}
		//CR 2316, DR 2723 
		private void lvInstitutions_ItemCheck(object sender, System.Windows.Forms.ItemCheckEventArgs e)
		{
			if(this.cbIncludesInstitutions.Checked == false && _refreshing == false)
			{
				if(e.CurrentValue == CheckState.Checked)
				{
					e.NewValue = CheckState.Checked;
				}
				else
				{
					e.NewValue = CheckState.Unchecked;
				}
				return;
			}

			if (!_refreshing)
			{
				System.DateTime currentDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime(this.BoundDivision.DivisionCode);

				DataRow dr = (DataRow)this.lvInstitutions.Items[e.Index].Tag;
				string divCode = dr[DivTable.DivisionCode].ToString();
				if (e.NewValue == CheckState.Checked)
				{
					System.Data.DataRow [] dtRows = this._assocInstitutions.Select(InstitutionTable.InstitutionDivisionCode + " = '" + dr[DivTable.DivisionCode].ToString() + "'");
					//Check if the newly checked item is already _assocInstitutions in the assoiciated divisions list, if not then add it to the datatable.
					if(dtRows.Length<=0)
					{	
						System.Data.DataRow dtNewRow = this._assocInstitutions.NewRow();
						dtNewRow[InstitutionTable.AssociatedInstitutionGuid] = System.Guid.NewGuid();
						dtNewRow[InstitutionTable.InstitutionDivisionCode] = dr[DivTable.DivisionCode];
						dtNewRow[InstitutionTable.DivisionCode] = this.BoundDivision.DivisionCode;
						dtNewRow[InstitutionTable.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
						dtNewRow[InstitutionTable.LastUpdateUser] = Common.LogonUser.LogonUserName;
						dtNewRow[InstitutionTable.LastUpdateDate] = currentDate;
						dtNewRow[InstitutionTable.LastUpdateFunctionId] = Common.UpdateFunction.MUC02_EditDivisionParameters;
						this._assocInstitutions.Rows.Add(dtNewRow);
					}
					else
					{
						dtRows[0][InstitutionTable.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
						dtRows[0][InstitutionTable.LastUpdateUser] = Common.LogonUser.LogonUserName;
						dtRows[0][InstitutionTable.LastUpdateDate] = currentDate;
						dtRows[0][InstitutionTable.LastUpdateFunctionId] = Common.UpdateFunction.MUC02_EditDivisionParameters;
					}
				}
				else
				{
					System.Data.DataRow [] dtRows = _assocInstitutions.Select(InstitutionTable.InstitutionDivisionCode + " = '" + dr[DivTable.DivisionCode].ToString() + "'");
					if(dtRows.Length>0)
					{
						if(dtRows[0].RowState == System.Data.DataRowState.Added)
						{
							_assocInstitutions.Rows.Remove(dtRows[0]);
						}
						else
						{
							dtRows[0][InstitutionTable.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
							dtRows[0][InstitutionTable.LastUpdateUser] = Common.LogonUser.LogonUserName;
							dtRows[0][InstitutionTable.LastUpdateDate] = currentDate;
							dtRows[0][InstitutionTable.LastUpdateFunctionId] = Common.UpdateFunction.MUC02_EditDivisionParameters;
						}
					}
				}
			}
		
			this.BoundDivision.AssociatedInstitutions = this._assocInstitutions;
			//CR 2345
			if (!_refreshing)
			{
				OnDataChanged( true );
			}
		}

		//CR 2316, DR 2723 
		private void lvInstitutions_ColumnClick(object sender, System.Windows.Forms.ColumnClickEventArgs e)
		{
			ColumnClickSorter.ColumnClickHandler( sender, e, true, ColumnClickSorter.ColumnSortType.String );
		}

		
	}
}